//==============================================================================
// Project:		Wealth Transfers and their economic effects
// File name:	Figures 2.1 2.2 B.6
// Objective: 	Generate the data underpinning figures 2.1, 2.2 and B.6
//
//==============================================================================

clear all

************************
*** Data import ***
************************

cd "XXXX" //set content directory here

use combined_filtered_restricted.dta, replace

merge m:1 xwaveid wave using bequest_from_HH_death.dta //flag for inheritances due to HH deaths
drop if _merge == 2
drop _merge

merge m:1 wave using CPI.dta //CPI data
drop if _merge == 2
drop _merge 

xtset xwaveid wave


************************
*** Create variables ***
************************

replace oifinha = 0 if oifinha == -10 | oifinha == -1 //inheritances
replace oifinha = . if oifinha < 0
gen bequest = oifinha
replace bequest = 0 if bequest > 0 & bequest != . & bequest_from_HH_death == 1 //filter out inheritances due to deaths from same HH, as no wealth transfer at the HH level in these cases
 
gen net_worth = hwnwip - hwnwin //household net worth

rename wlebf weight_2 //weights
rename wlefj weight_6
rename wlejn weight_10
rename wlenr weight_14
gen weight = weight_2 if wave == 2
replace weight = weight_6 if wave == 6
replace weight = weight_10 if wave == 10
replace weight = weight_14 if wave == 14

codebook weight if wave == 2 | wave == 6 | wave == 10 | wave == 14

gen child = 1 if hgage < 15 //generate equivalisation scale (OECD modified)
gen adult = 1 if hgage >= 15
egen total_child = total(child), by(hhrhid wave)
egen total_adult = total(adult), by(hhrhid wave)
gen equiv_scale = total_child*0.3 + total_adult*0.5 + 0.5 //0.3 for children, 1 for first adult and 0.5 for all subsequent adults

egen total_bequest = total(bequest), by(hhrhid wave)
gen bequest_e = total_bequest * 100 / (equiv_scale*CPI) //real equivalised inheritances

gen net_worth_e = net_worth * 100 / (equiv_scale*CPI) //real equivalised net worth

gen net_worth_bequest_e = net_worth_e + F2.bequest_e + F3.bequest_e + F4.bequest_e

gen F2_4_bequest_e = F2.bequest_e + F3.bequest_e + F4.bequest_e

gen F2_4_bequest_e_positive = 0
replace F2_4_bequest_e_positive = 1 if F2_4_bequest_e > 0 //flag for an inheritance

gen flag = 1

drop if net_worth_bequest_e == . //drop cases where the person not in each wave, due to death or moving abroad (the weights don't adjust for this)

**************************
*** Calculate quintiles **
**************************

*Lifecycle-adjusted quintiles
gen net_worth_quin_1 = .
local wealth_waves 2 6 10 14
foreach x of local wealth_waves{
	forvalues y = 0/100{
		capture noisily xtile net_worth_quin_1_`x'_`y' = net_worth_e [pweight = weight_`x'] if wave == `x' & hgage == `y', nq(5)
		capture noisily replace net_worth_quin_1 = net_worth_quin_1_`x'_`y' if wave == `x' & hgage == `y'
	}
}
svyset [pweight = weight] // note that strata(xhhstrat) cannot be used because of stratum with single sampling unit
svy: mean F2_4_bequest_e if (wave == 2 | wave == 6 | wave == 10 | wave == 14) & F2_4_bequest_e > 0, over(net_worth_quin_1) //figure B.6, panel b (note that CIs are estimated below, drawing on the standard errors computed here)
matrix mean_positive_LA = r(table)'
svyset [pweight = weight], strata(xhhstrat) psu(xhhraid) // see https://melbourneinstitute.unimelb.edu.au/assets/documents/hilda-bibliography/hilda-technical-papers/Hilda-Standard-Errors-2.08-Technical-paper-RVSD.pdf
svy: ratio F2_4_bequest_e_positive flag if (wave == 2 | wave == 6 | wave == 10 | wave == 14), over(net_worth_quin_1) //figure B.6, panel a
matrix share_positive_LA = r(table)'
svy: mean F2_4_bequest_e if (wave == 2 | wave == 6 | wave == 10 | wave == 14), over(net_worth_quin_1) //figure B.6, panel c
matrix mean_LA = r(table)' 
svy: ratio F2_4_bequest_e net_worth_e if (wave == 2 | wave == 6 | wave == 10 | wave == 14), over(net_worth_quin_1) //figure B.6, panel d
matrix relative_LA = r(table)'

*Standard quintiles
gen net_worth_quin_2 = .
local wealth_waves 2 6 10 14
foreach x of local wealth_waves{
    capture noisily xtile net_worth_quin_2_`x' = net_worth_e [pweight = weight_`x'] if wave == `x', nq(5)
	replace net_worth_quin_2 = net_worth_quin_2_`x' if wave == `x'
}

svyset [pweight = weight], // note that strata(xhhstrat) cannot be used because of stratum with single sampling unit
svy: mean F2_4_bequest_e if (wave == 2 | wave == 6 | wave == 10 | wave == 14) & F2_4_bequest_e > 0, over(net_worth_quin_2) //figure 2.1, panel b
matrix mean_positive = r(table)'
svyset [pweight = weight], strata(xhhstrat) psu(xhhraid)
svy: ratio F2_4_bequest_e_positive flag if (wave == 2 | wave == 6 | wave == 10 | wave == 14), over(net_worth_quin_2) //figure 2.1, panel a
matrix share_positive = r(table)'
svy: mean F2_4_bequest_e if (wave == 2 | wave == 6 | wave == 10 | wave == 14), over(net_worth_quin_2)
matrix mean = r(table)'
svy: mean net_worth_e if (wave == 2 | wave == 6 | wave == 10 | wave == 14), over(net_worth_quin_2)
svy: ratio F2_4_bequest_e net_worth_e if (wave == 2 | wave == 6 | wave == 10 | wave == 14), over(net_worth_quin_2) //figure 2.2
matrix relative = r(table)'

************************************************
*** Lifecycle-adjusted quintiles for charting **
************************************************
clear

svmat mean_positive_LA
rename mean_positive_LA1 b_mean_positive_LA
drop mean*
svmat mean_positive_LA
gen b_mean_positive_LA_95CI = mean_positive_LA2 * 1.96
drop mean*
svmat share_positive_LA
rename share_positive_LA1 b_share_positive_LA
replace b_share_positive_LA = 100 * b_share_positive_LA
drop share*
svmat share_positive_LA
gen b_share_positive_LA_95CI = 100 * share_positive_LA2 * 1.96
drop share*
svmat mean_LA
rename mean_LA1 b_mean_LA
drop mean*
svmat mean_LA
gen b_mean_LA_95CI = mean_LA2 * 1.96
drop mean*
svmat relative_LA
rename relative_LA1 b_relative_LA
replace b_relative_LA = 100 * b_relative_LA
drop relative*
svmat relative_LA
gen b_relative_LA_95CI = 100 * relative_LA2 * 1.96
drop relative*

export delimited LA_quintiles_for_charting_pooled_3.csv, replace

************************************************
*** Standard quintiles for charting **
************************************************
clear

svmat mean_positive
rename mean_positive1 b_mean_positive
drop mean*
svmat mean_positive
gen b_mean_positive_95CI = mean_positive2 * 1.96
drop mean*
svmat share_positive
rename share_positive1 b_share_positive
replace b_share_positive = 100 * b_share_positive
drop share*
svmat share_positive
gen b_share_positive_95CI = 100 * share_positive2 * 1.96
drop share*
svmat mean
rename mean1 b_mean
drop mean*
svmat mean
gen b_mean_95CI = mean2 * 1.96
drop mean*
svmat relative
rename relative1 b_relative
replace b_relative = 100 * b_relative
drop relative*
svmat relative
gen b_relative_95CI = 100 * relative2 * 1.96
drop relative*

export delimited quintiles_for_charting_pooled_3.csv, replace




















